/*
   Backprojection for imaging with photon density wave.
   2/23/1998, LW.
 */

#include <stdio.h>
#include <math.h>
#include <float.h>
#include "nrutil.h"
void	svdcmp(float **a, int m, int n, float w[], float **v);

#define SQ(x) ((x)*(x))

#ifndef PI
#define PI 		3.141592653589793
#endif

typedef struct {
  double	r, i;
} complex;

typedef struct {
  float	f;		/* frequency [Hz]. */
  float	c;		/* speed of light in medium [cm/s]. */
  float	h;		/* grid size [cm], dx = dy = dz = h. */
  int	N;		/* NxN grid. */
  int	gap;		/* gap between src/det & boundary [grid]. */
  float	mua0;		/* mua of background [/cm]. */
  float	mus0;		/* mus' of background [/cm]. */

  int	obj_x, obj_y, obj_size;	/* object location and size in grid. */
  float	dmua;		/* delta mua of object [/cm]. */
} ParamStru;

typedef struct {
  float	*VecUsc;		/* vector U. Real part 1st. */
  float	**MatW;		/* matrix W: weight. Real part 1st. */
  float	*VecMua;		/* vector mua. */
} ArrayStru;


complex
C(double rl, double im)
{
  complex     c;

  c.r = rl;
  c.i = im;
  return (c);
}


double	Re(complex C)
{
  return (C.r);
}


double	Im(complex C)
{
  return (C.i);
}


double	CAbs(complex C)
{
  return (sqrt(C.r * C.r + C.i * C.i));
}


complex
CAdd(complex C1, complex C2)
{
  complex     c;

  c.r = C1.r + C2.r;
  c.i = C1.i + C2.i;
  return (c);
}


complex
CSub(complex C1, complex C2)
{
  complex     c;

  c.r = C1.r - C2.r;
  c.i = C1.i - C2.i;
  return (c);
}


/* (a + ib)(c + id) = ac-bd + i(bc+ad) */
complex
CMul(complex C1, complex C2)
{
  complex     c;

  c.r = C1.r * C2.r - C1.i * C2.i;
  c.i = C1.r * C2.i + C1.i * C2.r;
  return (c);
}


/* (a + ib)/(c + id) = (a+ib)(c-id)/(c^2+d^2) */
complex
CDiv(complex C1, complex C2)
{
  double	temp;
  complex     c;

  temp = 1 / (C2.r * C2.r + C2.i * C2.i);
  c.r = (C1.r * C2.r + C1.i * C2.i) * temp;
  c.i = (C1.i * C2.r - C1.r * C2.i) * temp;
  return (c);
}


complex
CC(complex C)
{
  complex     ctemp;

  ctemp.r = C.r;
  ctemp.i = -C.i;
  return (ctemp);
}


complex
CSqrt(complex C)
{
  double	amp, ang;
  complex     ctemp;

  amp = sqrt(CAbs(C));
  ang = 0.5 * atan2(Im(C), Re(C));
  ctemp.r = amp * cos(ang);
  ctemp.i = amp * sin(ang);
  return (ctemp);
}


/* Calculate exp(C).
 */
complex
CExp(complex C)
{
  double	amp, ang;
  complex     ctemp;

  amp = exp(Re(C));
  ang = Im(C);
  ctemp.r = amp * cos(ang);
  ctemp.i = amp * sin(ang);
  return (ctemp);
}


/****************************************************************
 *  Report error message to stderr, then exit the program
 *  with signal 1.
 ****/
void	RepError(char *ErrorText)
{
  fprintf(stderr, "%s\n", ErrorText);
  fprintf(stderr, "...now exiting to system...\n");
  exit(1);
}


void	ShowProg(void)
{
  puts("");
  puts("Back Projection of Photon Density Wave");
  puts("Lihong Wang, Ph.D.");
  puts("Texas A&M University");
  puts("Version date: 2/23/1998");
  puts("");
}


void	SetParam(ParamStru *par)
{
  par->f = 200E6;
  par->c = 3E10 / 1.37;
  par->h = 0.2;
  par->N = 20;			/* number of voxels in each direction. */
  par->gap = 3;
  par->mua0 = 0.1;
  par->mus0 = 10;

  par->obj_x = 8;
  par->obj_y = 10;
  par->obj_size = 5;
  par->dmua = 0.01;
}


void	AllocArrays(ParamStru par, ArrayStru *arr)
{
  int	num_rows = 2 *par.N *par.N;	/* NxN measurements. */
  int	num_cols = par.N *par.N;

  arr->VecUsc = vector(1, num_rows);
  arr->VecMua = vector(1, num_cols);
  arr->MatW = matrix(1, num_rows, 1, num_cols);
}


complex
Green(ParamStru par, int ix, int iy)
{
  static int	once = 1;
  complex     k0, G;
  float	D = 1 / (3 *(par.mua0 + par.mus0));
  float	r = sqrt(SQ(ix *par.h) + SQ(iy *par.h));

  k0 = CSqrt(C(-par.mua0 / D, 2 * PI * par.f / (par.c * D)));
  k0 = CMul(k0, C(0, -r));	/* -ikr = k*(-r). */
  G = CExp(k0);
  G = CDiv(G, C(4 * PI * r, 0));

  if (once == 1) {
    /* printf("G = %e\n", G);
     */
    once = 2;
  }
  return (G);
}


complex
U0(ParamStru par, int ix, int iy)
{
  complex     U0;
  float	D = 1 / (3 *(par.mua0 + par.mus0));

  U0 = Green(par, ix, iy);
  U0 = CDiv(U0, C(par.c * D, 0));
  return (U0);
}


void	CalcUsc(ParamStru par, int ixs, int iys,
int ixd, int iyd, float *Usc_re, float *Usc_im)
{
  complex     kernal, Usc;
  int	ixo, iyo;
  float	D = 1 / (3 *(par.mua0 + par.mus0));

  Usc = C(0, 0);
  for (ixo = par.obj_x; ixo <= par.obj_x + par.obj_size; ixo++)
    for (iyo = par.obj_y; iyo <= par.obj_y + par.obj_size;
        iyo++) {
      kernal = CMul(U0(par, ixo - ixs, iyo - iys),
           		    Green(par, ixd - ixo, iyd - iyo));
      kernal = CMul(C(-par.dmua / D, 0), kernal);
      Usc = CAdd(Usc, kernal);
    }
  *Usc_re = Re(Usc) * pow(par.h, 3);
  *Usc_im = Im(Usc) * pow(par.h, 3);
}


void	CalcVectorUsc(ParamStru par, float *Usc)
{
  int	iys, iyd, isd, i_shift;

  i_shift = par.N * par.N;
  /* printf("Usc:\n");
   */
  for (iyd = 1; iyd <= par.N; iyd++) {
    for (iys = 1; iys <= par.N; iys++) {
      isd = iys + (iyd - 1) * par.N;
      CalcUsc(par, -par.gap, iys, par.N + par.gap,
           iyd,
          &Usc[isd], &Usc[isd + i_shift]);
      /* printf("%E\t%E\t", Usc[isd], Usc[isd + i_shift]);
       */
    }
  }
}


void	CalcW(ParamStru par, int ixs, int iys,
int ixd, int iyd,
int ixo, int iyo,
float *W_re, float *W_im)
{
  complex     W;
  float	D = 1 / (3 *(par.mua0 + par.mus0));

  W = CMul(U0(par, ixo - ixs, iyo - iys), Green(par, ixd - ixo,
       iyd - iyo));
  *W_re = -Re(W) * pow(par.h, 3) / D;
  *W_im = -Im(W) * pow(par.h, 3) / D;
}


void	CalcMatrixW(ParamStru par, float **W)
{
  int	iys, iyd, ixo, iyo;	/* indices to src, det, obj. */
  int	isd, iob;		/* combined indices. */
  int	i_shift;

  i_shift = par.N * par.N;
  /* printf("\nW:\n");
   */
  for (iyd = 1; iyd <= par.N; iyd++)
    for (iys = 1; iys <= par.N; iys++)
      for (iyo = 1; iyo <= par.N; iyo++)
	for (ixo = 1; ixo <= par.N; ixo++) {
	  isd = iys + (iyd - 1) * par.N;
	  iob = ixo + (iyo - 1) * par.N;
	  CalcW(par, -par.gap, iys, par.N +
	      par.gap, iyd, ixo, iyo,
	      &W[isd][iob], &W[isd + i_shift][iob]);
	  /* printf("%e\t%e\t", W[isd][iob], W[isd+i_shift][iob]);
	   */
	}
}


void	CalcVectorMua(ParamStru par, ArrayStru *arr)
{
  int	m, n;
  float	*w, **v;
  int	i, j, k;

  m = 2 * par.N * par.N;
  n = par.N * par.N;
  w = vector(1, n);
  v = matrix(1, n, 1, n);
  svdcmp(arr->MatW, m, n, w, v);
  for (i = 1; i <= n; i++) {
    arr->VecMua[i] = 0;
    for (j = 1; j <= n; j++)
      for (k = 1; k <= m; k++)
	arr->VecMua[i] += v[i][j] / w[j] * arr->MatW[k][j] *
	    arr->VecUsc[k];
  }
  free_vector(w, 1, n);
  free_matrix(v, 1, n, 1, n);
}


void	FPrintMu(ParamStru par, float *mua)
{
  FILE       * fp;
  int	ix, iy, ixy;

  fp = fopen("bp.im", "w");
  for (iy = 1; iy <= par.N; iy++) {
    for (ix = 1; ix <= par.N; ix++) {
      ixy = ix + (iy - 1) * par.N;
      if (ix < par.N)
	fprintf(fp, "%.5E\t", par.mua0 + mua[ixy]);
      else
	fprintf(fp, "%.5E\n", par.mua0 + mua[ixy]);
    }
  }
  fclose(fp);
}


void	main(int argc, char *argv[])
{
  ParamStru   params;
  ArrayStru   arrays;

  ShowProg();

  SetParam(&params);
  AllocArrays(params, &arrays);
  CalcVectorUsc(params, arrays.VecUsc);
  printf("Done Calc Usc\n");
  CalcMatrixW(params, arrays.MatW);
  printf("Done Calc W\n");
  CalcVectorMua(params, &arrays);
  printf("Done Calc Mua\n");
  FPrintMu(params, arrays.VecMua);
}


