%Program File for "Theory Driven Bias in Ideal Point Estimates - A Monte
%Carlo Study - 2009-01-07
%
%This program computes one-dimensional heckman-snyder ideal points from
%a roll call matrix
%
%Takes the following inputs
%rollcalls - a T by N roll call matrix, where N is the number of legislators
%            all votes must be either 1 (yea) or 0 (nay). no abstention.
%names     - numerical identity indices for each legislator
%leftmost  - a row vector with three columns containing the numerical
%            identity index, ideal point, and party of the leftmost legislator
%            the estimated ideal point will be pinned at the true value
%rightmost - same as ``leftmost'' for the rightmost legislator
%
%This program returns an (N+1) length column vector. The first row
%is the classification success, and the remaining rows are the estimated
%ideal points

function [estimate_idl] = estimate_idl(rollcalls, names, leftmost, rightmost);

n=length(names); %number legislators

%rotate matrix from TxN to NxT;
votes=rollcalls';

    %-------------- BEGIN FACTOR ANALYSIS ESTIMATION -------------
    
        dims=size(votes);
    
        %COMPUTE RAW ONE DIMENSIONAL HECKMAN SNYDER ESTIMATES;
        means=repmat(sum(votes)/n,n,1);         %mean yeas per roll call, matrix form
        xcenter=votes-means;                    %subtract mean yeas from actual rolls
        [U,S,V]=svd(xcenter);                   %singular value decomposition
        hs_coord_raw=U(:,1);                    %raw scores
        
        %COMPUTE CLASSIFICATION SUCCESS OF 1-D MODEL;
        xcenter_1d=U(:,1:1)*S(1:1,1:1)*V(:,1:1)';
        voteprobs_1d=xcenter_1d+means;
        votes_1d=round(voteprobs_1d);
        class_success=sum(sum(1*(votes==votes_1d)))/(dims(1)*dims(2));
        

        %PERFORM SCALING ANR ROTATION;

        %compute estimated idl pt of leftmost legislator
        hs_coord_leftmost=hs_coord_raw(names==leftmost(1));
        %compute estimated idl pt of rightmost legislator
        hs_coord_rightmost=hs_coord_raw(names==rightmost(1));
        %compute linear function which will place the leftmost and
        %rightmost legislators at their true values
        m=(leftmost(2)-rightmost(2))/(hs_coord_leftmost-hs_coord_rightmost);
        b=leftmost(2)-m*hs_coord_leftmost;
        %rotate and scale ideal points
        hs_coord=hs_coord_raw*m+b;
        
        %output
        estimate_idl=[class_success;hs_coord];

