%Program File for "Theory Driven Bias in Ideal Point Estimates - A Monte
%Carlo Study'' - 2009-01-07
%
%Generates three plots from simulation data with normal shocks
%
%1 - mean correllation coefficient for deadzone legislators across
%simulation sets for party cartel and median proposer models
%
%2 -  mean estimated ideal points for sigma=0
%3 -  mean estimated ideal points for sigma=.4

clear

%----------------READ IN TRUE IDEAL POINTS, COMPUTE PIVOTS----------------;
legislators_data=dlmread('outnormal\legislators');
legislators=legislators_data(:,:);
n=length(legislators);

pivots=calculate_pivots(legislators);

med=pivots(8);
majparty_med=pivots(6);
r_blockzone=majparty_med;

blockzone_leg=(legislators(:,2)>=med)&(legislators(:,2)<=r_blockzone);
%-------------------------------------------------------------------------;


%--------------------SETUP SIMULATION PERMUTATIONS------------------------;
s_iter=16;
startpt=0;
endpt=.60;

for s=1:1:s_iter; sigma(s)=startpt+(s-1)*(endpt-startpt)/max((s_iter-1),1); end;
%-------------------------------------------------------------------------;


%---COMPUTE MEAN CORRELATION COEFFICIENTS AND ESTIMATED IDL PTS-----------;
mean_idls_PC=zeros(n,s_iter);

for s=1:1:16;

    %read in party cartel data;
    dataPC=dlmread(strcat('outnormal\outputPC_',int2str(s)));
    dataPC=dataPC(:,5:end); %cut off first four columns
    
    %read in median proposer data;   
    dataMV=dlmread(strcat('outnormal\outputMV_',int2str(s)));
    dataMV=dataMV(:,5:end); %cut off first four columns
    
    %compute mean ideal points
    mean_idls_PC(:,s)=mean(dataPC)';
    mean_idls_MV(:,s)=mean(dataMV)';
    
    
    %select out deadzone legislators
    idlpts_BZ_PC=dataPC(:,blockzone_leg);
    idlpts_BZ_MV=dataMV(:,blockzone_leg);
    idlpts_BZ=legislators(blockzone_leg,2);
    
    %mechanical fix for errorless case so correlation is 0 instead of undefined 
    if (s==1)
        idlpts_BZ_PC(:,1)=idlpts_BZ_PC(:,1)+.001;
    end;
    
    %blockzone legislator correlations per run
    corrs_BZ_PC=corr(idlpts_BZ_PC',idlpts_BZ);
    corrs_BZ_MV=corr(idlpts_BZ_MV',idlpts_BZ);
    
    %mean correlation of blockzone legislators
    mean_corrs_BZ_PC(s)=mean(corrs_BZ_PC);
    mean_corrs_BZ_MV(s)=mean(corrs_BZ_MV);

    %correlation of true values with means, blockzone legislators
    corr_ofmeans_BZ_PC(s)=corr(mean_idls_PC(blockzone_leg,s),idlpts_BZ);
    corr_ofmeans_BZ_MV(s)=corr(mean_idls_MV(blockzone_leg,s),idlpts_BZ);

    
    %-----------compute same statistics for all legislators----------;
    
    %all legislator correllations per run
    corrs_PC=corr(dataPC',legislators(:,2));
    corrs_MV=corr(dataMV',legislators(:,2));
    
    %mean correllation all
    mean_corrs_PC(s)=mean(corrs_PC);
    mean_corrs_MV(s)=mean(corrs_MV);

    %correllation with means, all
    corr_ofmeans_PC(s)=corr(mean_idls_PC(:,s),legislators(:,2));
    corr_ofmeans_MV(s)=corr(mean_idls_MV(:,s),legislators(:,2));

end;
%-------------------------------------------------------------------------;


%------------------------FIRST FIGURE, CORRELATIONS---------------------;
figure(2);
hold on;

%plot correlations for party cartel game
scatter(sigma',mean_corrs_BZ_PC',40,'g','filled');
%plot correlations for median proposer game
scatter(sigma',mean_corrs_BZ_MV',40,'b','*');

%formatting
legend('Run 1 (Party Cartel)','Run 2 (Median Proposer)');
set(gca,'Xlim',[0 endpt])
xlabel('\sigma','fontsize',14)
set(gca,'Ylim',[.5 1])
ylabel('Correlation','fontsize',14)
set(gca,'fontsize',14)

hold off;
saveas(2,'outnormal\Mean_Correllation.pdf');

%output figure data in CSV form
dlmwrite('outnormal\mean_corrs_BZ_PC',[sigma' mean_corrs_BZ_PC']);
dlmwrite('outnormal\mean_corrs_BZ_MV',[sigma' mean_corrs_BZ_MV']);
%-------------------------------------------------------------------------;


%--------SECOND FIGURE, MEAN IDL PTS AT SIGMA=.4--------------------------;
figure(3);

hold on;
%plot mean estimates for party cartel game
scatter(legislators(:,2),mean_idls_PC(:,11))
%plot mean estimates for median proposer game
scatter(legislators(:,2),mean_idls_MV(:,11),'*')
    
%demarcate deadzone 
line([med med],[-2 2]);
line([r_blockzone r_blockzone],[-2 2]); 

%formatting
legend('Run 1 (Party Cartel)','Run 2 (Median Proposer)');
xlabel('True Ideal Point','fontsize',14)
ylabel('Mean Estimated Ideal Point','fontsize',14)
set(gca,'Xlim',[-1 1.5])
set(gca,'Ylim',[-1 1.5])
set(gca,'fontsize',14)
hold off;
saveas(3,'outnormal\TrueVEstimated_p4.pdf');

%output data in CSV form
dlmwrite('outnormal\TrueVEstimated_p4_PC',[legislators(:,2) mean_idls_PC(:,11)]);
dlmwrite('outnormal\TrueVEstimated_p4_MV',[legislators(:,2) mean_idls_MV(:,11)]);
%-------------------------------------------------------------------------;


%--------THIRD FIGURE, MEAN IDL PTS AT SIGMA=0--------------------------;
figure(4);
hold on;
%plot mean estimates for party cartel game
scatter(legislators(:,2),mean_idls_PC(:,1))
%plot mean estimates for median proposer game
scatter(legislators(:,2),mean_idls_MV(:,1),'*')

%demarcate deadzone 
line([med med],[-2 2]);
line([r_blockzone r_blockzone],[-2 2]); 

%formatting
legend('Run 1 (Party Cartel)','Run 2 (Median Proposer)');
xlabel('True Ideal Point','fontsize',14)
ylabel('Mean Estimated Ideal Point','fontsize',14)
set(gca,'Xlim',[-1 1.5])
set(gca,'Ylim',[-1 1.5])
set(gca,'fontsize',14)
hold off;
saveas(4,'outnormal\TrueVEstimated_0.pdf');

%output data in CSV form
dlmwrite('outnormal\TrueVEstimated_0_PC',[legislators(:,2) mean_idls_PC(:,1)]);
dlmwrite('outnormal\TrueVEstimated_0_MV',[legislators(:,2) mean_idls_MV(:,1)]);
%-------------------------------------------------------------------------;

