%Program File for "Theory Driven Bias in Ideal Point Estimates - A Monte
%Carlo Study'' - 2009-01-07
%
%Computes and plots densities of the estimated blockzone width for each 
%set of simulated legislative sessions in the normal shocks case

clear

%---------------------SET PARAMETERS-------------------------------------;
z_iter=2000; %number of sessions in each set
percent=.95; %area of density to shade around the mean
h=.005; %bandwidth parameter for kernel density estimate
x=[-.1:.0001:.75]; %values for ksdensity estimate
x=round(x*10000)/10000; %required for reasons i cannot fathom
%-------------------------------------------------------------------------;


%----------------READ IN TRUE IDEAL POINTS, COMPUTE PIVOTS----------------;
legislators_data=dlmread('outnormal\legislators');
legislators=sortrows(legislators_data,2);
n=length(legislators);

pivots=calculate_pivots(legislators);

med_index=pivots(7);
med=pivots(8);
majparty_med_index=pivots(5);
majparty_med=pivots(6);

r_blockzone=majparty_med;

blockzone_leg=(legislators(:,2)>=med)&(legislators(:,2)<=r_blockzone);
%-------------------------------------------------------------------------;


%--------------------SETUP SIMULATION PERMUTATIONS------------------------;
s_iter=16;
startpt=0;
endpt=.60;

for s=1:1:s_iter; sigma(s)=startpt+(s-1)*(endpt-startpt)/max((s_iter-1),1); end;
%-------------------------------------------------------------------------;


%--------------------PLACEHOLDER VARIABLES--------------------------------;
BZ_width_PC=zeros(z_iter,s_iter);
BZ_width_MV=zeros(z_iter,s_iter);

BZ_width_density_PC=zeros(length(x),s_iter);
BZ_width_density_MV=zeros(length(x),s_iter);

BZ_PRegion_PC=zeros(2,s_iter);
%-------------------------------------------------------------------------;


%----------------COMPUTE DENSITIES FOR PLOTS------------------------------;
%iterate through simulation sets
for s=1:1:s_iter;

    %read in party cartel data;
    dataPC=dlmread(strcat('outnormal\outputPC_',int2str(s)));
    dataPC=dataPC(:,5:end); %cut off first four columns
    
    %read in median proposer data;    
    dataMV=dlmread(strcat('outnormal\outputMV_',int2str(s)));
    dataMV=dataMV(:,5:end); %cut off first four columns
    
    %compute estimated pivots in each simulated session;
    %compute estimated deadzone widths in each simulated session
    for i=1:1:z_iter;
        
        legislators_PC=[legislators(:,1) dataPC(i,:)' legislators(:,3)];
        legislators_PC=sortrows(legislators_PC,2);
        
        legislators_MV=[legislators(:,1) dataMV(i,:)' legislators(:,3)];
        legislators_MV=sortrows(legislators_MV,2);
        
        pivots_PC=calculate_pivots(legislators_PC);
        pivots_MV=calculate_pivots(legislators_MV);
        
        BZ_width_PC(i,s)=pivots_PC(6)-pivots_PC(8);
        BZ_width_MV(i,s)=pivots_MV(6)-pivots_MV(8);
                
    end;
    
    %compute kernel density estimates of deadzone width
    for i=1:1:length(x);  BZ_width_density_PC(i,s)=sum(normpdf((BZ_width_PC(:,s)-x(i))/h,0,1))/(h*z_iter);   end;
    for i=1:1:length(x);  BZ_width_density_MV(i,s)=sum(normpdf((BZ_width_MV(:,s)-x(i))/h,0,1))/(h*z_iter);   end;
    
    %compute percentage regions;
    BZ_PRegion_PC(:,s)=percent_bar(BZ_width_PC(:,s), x, BZ_width_density_PC(:,s), percent)';
      
end;
%-------------------------------------------------------------------------;


%----------COMPUTE AND OUTPUT MEAN ESTIMATED DEADZONE WIDTHS--------------;
BZ_width_PC_mean=mean(BZ_width_PC)';
BZ_width_MV_mean=mean(BZ_width_MV)';
dlmwrite('outnormal\BZ_width_PC_mean',[sigma' BZ_width_PC_mean]);
dlmwrite('outnormal\BZ_width_MV_mean',[sigma' BZ_width_MV_mean]);
%-------------------------------------------------------------------------;


%----------------GENERATE PLOTS FROM COMPUTED DENSITIES-------------;

%for s=1:1:s_iter;
for s=1:1:9;   %truncated for space

    subplot(3,3,s)

    %identify region to shade
    pregion=(x>=(BZ_PRegion_PC(1,s)-BZ_PRegion_PC(2,s))&(x<=(BZ_PRegion_PC(1,s)+BZ_PRegion_PC(2,s))));
        
    hold on;
    
    %shade region
    c=area(x(pregion),BZ_width_density_PC(pregion,s));
    set(c(1),'FaceColor',[.8 .8 .8]);
    set(c(1),'EdgeColor',[1 1 1]);

    %plot density for party cartel game
    a=plot(x,BZ_width_density_PC(:,s),'r','LineWidth',1.5);
    
    %plot density for median proposer game
    b=plot(x,BZ_width_density_MV(:,s),'g--','LineWidth',1.5);

    %title and legend
    title(strcat('\sigma=',num2str(sigma(s))));
    legend([a b],'Run 1','Run 2');

    %place solid lines on axes
    line([0 0],ylim,'Color','k');
    line(xlim,[0 0],'Color','k');

    %set limits of x axis
    set(gca,'Xlim',[-.5 .65])
    
    hold off;
    
    
end;

%print to file
set(gcf,'PaperOrientation','landscape','PaperPosition', [0 0 11 7.75]);
print -dpdf 'outnormal\BZ_Densities_EstPiv.pdf'