/******************************************************************************
 *
 * simple_server.c
 * 
 * Original author: David Noblet
 * 
 * Description: 
 *   This is a small network server designed to accept a single connection, 
 *   read and write some data, and then exit.
 *
 *****************************************************************************/

// Standard C library includes
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>
#include <errno.h>
#include <unistd.h>

// Socket API includes
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

// Socket-related definitions
#define PROTO_DEFAULT 0

// Application-specific definitions
#define APP_NAME argv[0]
#define ARG_HOST 1
#define ARG_PORT 2
#define TOTAL_ARGS 2
#define LISTEN_QUEUE_SIZE 1
#define ANSWER_STRING "42\n"

// Application-defined function prototypes
static int check_args(int argc, char** argv);
static void print_usage(FILE* fd, const char* name);
static void print_error(FILE* fd, const char* name, const char* message);
static int get_listen_address(char** argv, struct sockaddr_in* sockaddr);
static int read_question(int fd, char* qbuf, int qlen);
static int write_answer(int fd, const char* answer);

// Application entry point
int main(int argc, char** argv)
{
    struct sockaddr_in sockaddr;
    int err, opt, sock_listen, sock_conn;
    
    // Check the user-supplied arguments
    if (check_args(argc, argv))
    {
        print_usage(stderr, APP_NAME);
        exit(1);
    }
    
    // Get the address & port from the user
    fprintf(stderr,"Getting listen address...\n");
    if((err = get_listen_address(argv, &sockaddr)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(-err));
        exit(1);
    }
    
    // Create a new TCP socket
    fprintf(stderr,"Creating TCP socket...\n");
    if((sock_listen = socket(AF_INET, SOCK_STREAM, PROTO_DEFAULT)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(errno));
        exit(1);
    }
    
    // Try to allow immediate reuse of the address
    opt = 1;
    setsockopt(sock_listen, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
    
    // Bind the socket to the supplied address and port
    fprintf(stderr,"Binding to address...\n");
    if(bind(sock_listen, (struct sockaddr*) &sockaddr, sizeof(sockaddr)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(errno));
        exit(1);
    }
    
    // Listen for incoming connections
    fprintf(stderr,"Set the socket to listen for incoming connections...\n");
    if(listen(sock_listen,LISTEN_QUEUE_SIZE) < 0)
    {
        print_error(stderr, APP_NAME, strerror(errno));
        exit(1);
    }
    
    // Accept a pending connection request
    fprintf(stderr,"Waiting to accept pending request...\n");
    if((sock_conn = accept(sock_listen, NULL, NULL)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(errno));
        exit(1);
    }
    
    // Read some data
    fprintf(stderr,"Waiting for data to read...\n");
    if((err = read_question(sock_conn, NULL, 0)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(-err));
        exit(1);
    }
    
    // Send a response
    fprintf(stderr,"Sending response...\n");
    if((err = write_answer(sock_conn, ANSWER_STRING)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(-err));
        exit(1);
    }
    
    // Clean up
    shutdown(sock_conn, SHUT_RDWR);
    close(sock_listen);
    
    return 0;
}

// Check application was passed appropriate arguments
static int check_args(int argc, char** argv)
{
    return (argc != TOTAL_ARGS+1);
}

// Print the application usage
static void print_usage(FILE* fd, const char* name)
{
    fprintf(fd, "Usage: %s %s\n", name, 
        "listen_address listen_port");
}

// Print an error message
static void print_error(FILE* fd, const char* name, const char* message)
{
    fprintf(fd, "%s: %s\n", name, message);
}

// Parse command-line arguments to get the listen address and port
static int get_listen_address(char** argv, struct sockaddr_in* sockaddr)
{
    int err = 0;
    memset(sockaddr,0,sizeof(struct sockaddr_in));
    sockaddr->sin_family = AF_INET;
    sockaddr->sin_port = htons(strtoul(argv[ARG_PORT], NULL, 10));
    if(errno)
    {
        return -errno;
    }
    if((err = inet_pton(AF_INET, argv[ARG_HOST], &sockaddr->sin_addr)) < 0)
    {
        return -err;
    }
    
    return err;
}

// Read until we hit a '?'; populate an optional buffer
static int read_question(int fd, char* qbuf, int qlen)
{
    int err = 0, idx = 0;
    char cbuf;
    
    // NOTE: Real applications should not read a single character at a time
    while((err = read(fd,&cbuf,1)) == 1)
    {
        if(qbuf)
        {
            qbuf[idx++] = cbuf;
            idx %= qlen;
        }
        
        if(cbuf == '?')
        {
            if (qbuf)
                qbuf[idx] = 0;
            return 0;
        }
    }
    
    return (errno == 0) ? -EINVAL : -errno;
}

// Write a string to the given descriptor
static int write_answer(int fd, const char* answer)
{
    int err = 0;
    int len;
    
    len = strlen(answer);
    
    if((err = write(fd, answer, len)) != len)
        return -errno;
    
    return 0;
}

