/******************************************************************************
 *
 * simple_client.c
 * 
 * Original author: David Noblet
 * 
 * Description: 
 *   This is a small network client designed to connect to a given server, 
 *   write some data, read a response (and print it), and then exit.
 *
 *****************************************************************************/

// Standard C library includes
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>
#include <errno.h>
#include <unistd.h>

// Socket API includes
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

// Socket-related definitions
#define PROTO_DEFAULT 0

// Application-specific definitions
#define APP_NAME argv[0]
#define ARG_HOST 1
#define ARG_PORT 2
#define TOTAL_ARGS 2
#define QUESTION_STRING \
    "What is the answer to life, the universe, and everything?\n"
#define LARGE_BUFFER_SIZE 1024

// Application-defined function prototypes
static int check_args(int argc, char** argv);
static void print_usage(FILE* fd, const char* name);
static void print_error(FILE* fd, const char* name, const char* message);
static int get_server_address(char** argv, struct sockaddr_in* sockaddr);
static int write_question(int fd, const char* answer);
static int read_answer(int fd, char* qbuf, int qlen);

// Application entry point
int main(int argc, char** argv)
{
    struct sockaddr_in sockaddr;
    int err, opt, sock_conn;
    char responsebuf[LARGE_BUFFER_SIZE];
    
    // Check the user-supplied arguments
    if (check_args(argc, argv))
    {
        print_usage(stderr, APP_NAME);
        exit(1);
    }
    
    // Get the server address & port from the user
    fprintf(stderr,"Getting server address...\n");
    if((err = get_server_address(argv, &sockaddr)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(-err));
        exit(1);
    }
    
    // Create a new TCP socket
    fprintf(stderr,"Creating TCP socket...\n");
    if((sock_conn = socket(AF_INET, SOCK_STREAM, PROTO_DEFAULT)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(errno));
        exit(1);
    }
    
    // Try to allow immediate reuse of the address
    opt = 1;
    setsockopt(sock_conn, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
    
    // Connect to the server
    fprintf(stderr,"Connecting to the server...\n");
    if(connect(sock_conn, (struct sockaddr*) &sockaddr, sizeof(sockaddr)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(errno));
        exit(1);
    }
    
    // Send a response
    fprintf(stderr,"Writing some data...\n");
    if((err = write_question(sock_conn, QUESTION_STRING)) < 0)
    {
        print_error(stderr, APP_NAME, strerror(-err));
        exit(1);
    }
    
    // Read some data
    fprintf(stderr,"Reading the response...\n");
    if((err = read_answer(sock_conn, responsebuf, sizeof(responsebuf))) < 0)
    {
        print_error(stderr, APP_NAME, strerror(-err));
        exit(1);
    }
    
    // Clean up
    close(sock_conn);
    
    // Print answer to stdout
    fprintf(stdout, "Q: %s\nA: %s\n", QUESTION_STRING, responsebuf);
    
    return 0;
}

// Check application was passed appropriate arguments
static int check_args(int argc, char** argv)
{
    return (argc != TOTAL_ARGS+1);
}

// Print the application usage
static void print_usage(FILE* fd, const char* name)
{
    fprintf(fd, "Usage: %s %s\n", name, 
        "server_address server_port");
}

// Print an error message
static void print_error(FILE* fd, const char* name, const char* message)
{
    fprintf(fd, "%s: %s\n", name, message);
}

// Parse command-line arguments to get the server address and port
static int get_server_address(char** argv, struct sockaddr_in* sockaddr)
{
    int err = 0;
    memset(sockaddr,0,sizeof(struct sockaddr_in));
    sockaddr->sin_family = AF_INET;
    sockaddr->sin_port = htons(strtoul(argv[ARG_PORT], NULL, 10));
    if(errno)
    {
        return -errno;
    }
    if((err = inet_pton(AF_INET, argv[ARG_HOST], &sockaddr->sin_addr)) < 0)
    {
        return -err;
    }
    
    return err;
}

// Read until we hit EOF; populate an optional buffer
static int read_answer(int fd, char* qbuf, int qlen)
{
    int err = 0, idx = 0;
    char cbuf;
    
    // NOTE: Real applications should not read a single character at a time
    while((err = read(fd,&cbuf,1)) == 1)
    {
        if(qbuf)
        {
            qbuf[idx++] = cbuf;
            idx %= qlen;
        }
    }
    
    if(qbuf)
        qbuf[idx] = 0;
    
    return -errno;
}

// Write a string to the given descriptor
static int write_question(int fd, const char* answer)
{
    int err = 0;
    int len;
    
    len = strlen(answer);
    
    if((err = write(fd, answer, len)) != len)
        return -errno;
    
    return 0;
}

